' sap-script-print-trp.vbs - Prints TRP/DPI files from SAP ZPACK transaction
' Complete version handling both PowerPoint and PDF files

On Error Resume Next

' Set scripting mode to true
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting warnings in registry
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' Declare log file object and path
Dim logFile, logFilePath

' Initialize log file if path is provided
If WScript.Arguments.Count > 2 Then
    logFilePath = WScript.Arguments.Item(2)
    ' Try to open the log file in append mode (8)
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script start
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "Quantity: " & quantity
        LogToFile "========================================="
    End If
End If

' Function to log message to file
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Combined logging function that writes to both console and file
Sub LogMessage(message)
    WScript.Echo message
    LogToFile message
End Sub

' Get command line arguments
materialNumber = WScript.Arguments.Item(0)  ' Material number for ZPACK
quantity = 1                                ' Default to 1 copy
If WScript.Arguments.Count > 1 Then
    quantity = CInt(WScript.Arguments.Item(1)) ' Get quantity if provided
End If

' Initialize SAP connection
If Not IsObject(application) Then
   Set SapGuiAuto = GetObject("SAPGUI")
   Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If
If Not IsObject(session) Then
   Set session = connection.Children(0)
End If
If IsObject(WScript) Then
   WScript.ConnectObject session, "on"
   WScript.ConnectObject application, "on"
End If

' Handle security dialog
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    Do While timeout < 10
        If shell.AppActivate("SAP GUI Security") Then
            WScript.Sleep 500
            shell.SendKeys " "  ' Press space to check "Remember my decision"
            WScript.Sleep 200
            shell.SendKeys "{ENTER}"  ' Press Enter to click Allow
            dialogFound = True
            Exit Do
        End If
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' Function to check if PowerPoint is running
Function IsPowerPointRunning()
    Dim WshShell, oExec, line, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Run tasklist and look for PowerPoint processes
    On Error Resume Next
    Set oExec = WshShell.Exec("tasklist /FI ""IMAGENAME eq POWERPNT.EXE""")
    
    ' Read output
    Do While Not oExec.StdOut.AtEndOfStream
        line = oExec.StdOut.ReadLine()
        ' Check if PowerPoint process is listed
        If InStr(1, line, "POWERPNT.EXE", vbTextCompare) > 0 Then
            isRunning = True
            Exit Do
        End If
    Loop
    
    IsPowerPointRunning = isRunning
End Function

' Function to check if Adobe Reader/Acrobat is running
Function IsAdobeRunning()
    Dim WshShell, adobeProcesses, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Define array of possible Adobe process names
    adobeProcesses = Array("AcroRd32.exe", "Acrobat.exe", "AcroRd.exe", "AdobeAcrobatDC.exe", "reader_sl.exe")
    
    ' Check each process individually
    On Error Resume Next
    For Each process In adobeProcesses
        Dim cmd, oExec, output
        cmd = "tasklist /FI ""IMAGENAME eq " & process & """ /NH"
        Set oExec = WshShell.Exec(cmd)
        output = oExec.StdOut.ReadAll()
        
        ' If process is found in output, Adobe is running
        If InStr(output, process) > 0 Then
            LogMessage "DEBUG: Found Adobe process: " & process
            isRunning = True
            Exit For
        End If
    Next
    
    IsAdobeRunning = isRunning
End Function

' Add this function after your IsAdobeRunning function
Function IsEdgeRunning()
    Dim WshShell, oExec, line, isRunning
    Set WshShell = CreateObject("WScript.Shell")
    isRunning = False
    
    ' Check for Edge processes
    On Error Resume Next
    Set oExec = WshShell.Exec("tasklist /FI ""IMAGENAME eq msedge.exe"" /NH")
    If Not oExec.StdOut.AtEndOfStream Then
        line = oExec.StdOut.ReadAll()
        If InStr(line, "msedge.exe") > 0 Then
            isRunning = True
            LogMessage "DEBUG: Microsoft Edge detected"
        End If
    End If
    
    IsEdgeRunning = isRunning
End Function

' Improved PowerPoint handling function with absolutely one tab only
Sub HandlePowerPoint(shell, quantity)
    ' Clear any pending keys first, just in case
    Err.Clear
    
    ' Send CTRL+P to open print dialog
    shell.SendKeys "^p"
    WScript.Sleep 2000 ' Wait for print dialog to open

    ' Single tab only to reach copies field (make sure this doesn't get modified)
    LogMessage "DEBUG: Sending single tab to PowerPoint dialog"
    shell.SendKeys "{TAB}"
    LogMessage "DEBUG: Tab sent"
    WScript.Sleep 1500 ' Longer wait to ensure focus is set
    
    ' Clear field and set quantity directly
    shell.SendKeys "{HOME}+{END}{DEL}"
    WScript.Sleep 800
    shell.SendKeys quantity
    WScript.Sleep 1500 ' Much longer wait to ensure quantity is entered
    
    ' Press Enter to print (Enter key, not tilde)
    shell.SendKeys "{ENTER}"
    WScript.Sleep 5000 ' Wait for print to start
    
    ' Force close PowerPoint using taskkill (the most reliable approach)
    On Error Resume Next
    Set oShell = CreateObject("WScript.Shell")
    oShell.Run "taskkill /IM POWERPNT.EXE /F", 0, True
    On Error GoTo 0
    
    WScript.Sleep 1500 ' Give it more time to complete
End Sub

' Enhanced PDF handling function for Adobe Acrobat Reader - NO ALT+TAB
Sub HandlePdf(shell, quantity, docType)
    ' Clear any pending keys first
    Err.Clear
    
    ' Wait for PDF to fully load
    LogMessage "BASIC-DEBUG: Waiting for PDF to fully load..."
    WScript.Sleep 3000
    
    ' --- Window Activation - ONLY direct window title methods ---
    LogMessage "BASIC-DEBUG: Attempting to activate Adobe Reader window..."
    Dim activated : activated = False
    
    ' Try ONLY reliable window activation methods (NO ALT+TAB)
    If shell.AppActivate("Adobe Acrobat Reader") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Adobe Acrobat Reader directly"
    ElseIf shell.AppActivate("Adobe Acrobat") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Adobe Acrobat directly"
    ElseIf shell.AppActivate("Acrobat") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated Acrobat directly"
    ElseIf shell.AppActivate(".pdf") Then
        activated = True
        LogMessage "BASIC-DEBUG: Activated window with .pdf in title"
    End If
    
    ' If not activated, just continue and try to send keys anyway
    ' But do NOT use ALT+TAB which can cause problems
    If Not activated Then
        LogMessage "BASIC-DEBUG: Could not activate window directly, will try to send keys anyway"
    End If
    
    ' Additional delay before sending print command
    WScript.Sleep 1000
    
    ' --- Printing ---
    ' Send Ctrl+P to open print dialog
    LogMessage "BASIC-DEBUG: Sending Ctrl+P to open print dialog"
    shell.SendKeys "^p"
    WScript.Sleep 2000  ' Allow more time for print dialog to open
    
    ' Try Alt+C for copies field (standard Adobe shortcut)
    LogMessage "BASIC-DEBUG: Sending Alt+C for copies field"
    shell.SendKeys "%c"
    WScript.Sleep 1000
    
    ' Clear field and enter quantity 
    LogMessage "BASIC-DEBUG: Setting quantity to " & quantity
    shell.SendKeys "{HOME}+{END}{DEL}"
    WScript.Sleep 800
    shell.SendKeys quantity
    WScript.Sleep 1500
    
    ' Press Enter to print
    LogMessage "BASIC-DEBUG: Sending Enter to print"
    shell.SendKeys "{ENTER}"
    WScript.Sleep 4000
    
    ' Force close Adobe using taskkill
    LogMessage "BASIC-DEBUG: Force closing Adobe"
    On Error Resume Next
    Set oShell = CreateObject("WScript.Shell")
    oShell.Run "taskkill /IM AcroRd32.exe /F", 0, True
    oShell.Run "taskkill /IM Acrobat.exe /F", 0, True
    On Error GoTo 0
    
    WScript.Sleep 1500
    LogMessage "BASIC-DEBUG: Adobe PDF handling completed for " & docType
End Sub

' Handle the Edge PDF printing case differently - print and then navigate back in SAP
Sub HandleEdgePdf(shell, quantity)
    ' Clear any pending keys first
    Err.Clear
    
    ' Wait for Edge to fully load the PDF
    WScript.Sleep 5000
    LogMessage "DEBUG: Starting Edge PDF print procedure"
    
    ' --- Window Activation ---
    Dim activated : activated = False
    
    ' Try different window titles to activate Edge
    If shell.AppActivate("Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated 'Microsoft Edge'"
    ElseIf shell.AppActivate(".pdf - Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated '.pdf - Microsoft Edge'"
    ElseIf shell.AppActivate("PDF Viewer - Microsoft Edge") Then
        activated = True
        LogMessage "DEBUG: Activated 'PDF Viewer - Microsoft Edge'"
    End If
    
    ' If still not activated, try a more generic approach
    If Not activated Then
        LogMessage "DEBUG: Unable to activate Edge window directly, trying broader approach"
        shell.AppActivate("Edge")
    End If
    
    ' --- Printing ---
    ' Send Ctrl+P to open print dialog
    LogMessage "DEBUG: Sending Ctrl+P to open print dialog in Edge"
    shell.SendKeys "^p"
    WScript.Sleep 3000  ' Wait longer for Edge's print dialog
    
    ' Tab multiple times to navigate to the copies field (adjust count as needed)
    LogMessage "DEBUG: Navigating to copies field in Edge print dialog"
    For i = 1 To 3  ' Edge usually needs about 7 tabs to reach copies field
        shell.SendKeys "{TAB}"
        WScript.Sleep 600
    Next
    
    ' Clear field and set quantity
    LogMessage "DEBUG: Setting quantity to " & quantity & " in Edge print dialog"
    shell.SendKeys quantity
    WScript.Sleep 1500
    
    ' Press Enter to print
    LogMessage "DEBUG: Sending Enter to print from Edge"
    shell.SendKeys "{ENTER}"
    WScript.Sleep 5000  ' Wait for print to start
    
    LogMessage "DEBUG: Finished Edge PDF print procedure, Edge will stay open"
    ' We don't need the EdgeWasUsed flag anymore
End Sub

' Function to process a document (either TRP or DPI)
Sub ProcessDocument(fileName, docType)
    ' Wait for the document viewer to open
    WScript.Sleep 3000
    
    ' Create shell for handling keyboard input
    Set shell = CreateObject("WScript.Shell")
    
    ' Determine file format by checking what's running - BASIC DEBUG FOR RELIABILITY
    LogMessage "BASIC-DEBUG: Starting process document checks"
    Dim isPowerPoint, isAdobe, isEdge
    isPowerPoint = IsPowerPointRunning()
    isAdobe = IsAdobeRunning()
    isEdge = IsEdgeRunning()
    
    ' Add direct console output for troubleshooting
    LogMessage "BASIC-DEBUG: PowerPoint detected: " & isPowerPoint
    LogMessage "BASIC-DEBUG: Adobe detected: " & isAdobe
    LogMessage "BASIC-DEBUG: Microsoft Edge detected: " & isEdge
    
    ' FIX: Prioritize document types in a reliable sequence
    If isPowerPoint Then
        LogMessage "BASIC-DEBUG: Using PowerPoint handler"
        HandlePowerPoint shell, quantity
        LogMessage "BASIC-DEBUG: PowerPoint document printed and closed"
    
    ' IMPORTANT CHANGE: Check Adobe BEFORE Edge
    ElseIf isAdobe Then
        LogMessage "BASIC-DEBUG: Using Adobe PDF handler"
        HandlePdf shell, quantity, docType
        LogMessage "BASIC-DEBUG: PDF document printed and closed"
    
    ' Only use Edge if Adobe is not running
    ElseIf isEdge Then
        LogMessage "BASIC-DEBUG: Using Edge PDF handler"
        HandleEdgePdf shell, quantity
        LogMessage "BASIC-DEBUG: Edge PDF document printed (browser left open)"
        
        ' Only navigate back for Edge
        LogMessage "BASIC-DEBUG: Navigating back in SAP since Edge stays open"
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    Else
        LogMessage "BASIC-DEBUG: No known application detected. Waiting longer..."
        WScript.Sleep 5000
        
        ' Retry detection
        isPowerPoint = IsPowerPointRunning()
        isAdobe = IsAdobeRunning()
        isEdge = IsEdgeRunning()
        
        ' Same priority in retry: PowerPoint, then Adobe, then Edge
        If isPowerPoint Then
            HandlePowerPoint shell, quantity
            LogMessage "BASIC-DEBUG: PowerPoint document printed and closed (delayed)"
        
        ElseIf isAdobe Then
            HandlePdf shell, quantity, docType
            LogMessage "BASIC-DEBUG: PDF document printed and closed (delayed)"
        
        ElseIf isEdge Then
            HandleEdgePdf shell, quantity
            LogMessage "BASIC-DEBUG: Edge PDF document printed (browser left open) (delayed)"
            
            LogMessage "BASIC-DEBUG: Navigating back in SAP since Edge stays open (delayed case)"
            session.findById("wnd[0]/tbar[0]/btn[3]").press
            session.findById("wnd[0]/tbar[0]/btn[3]").press
            session.findById("wnd[0]/tbar[0]/btn[3]").press
        Else
            LogMessage "BASIC-DEBUG: ERROR: No supported application detected for printing"
        End If
    End If
    
    ' Return success
    If docType = "TRP" Then
        LogMessage "SUCCESS: " & fileName
    Else
        LogMessage "DPI-SUCCESS"
    End If
End Sub

' Call HandleSecurityDialog before starting operations
HandleSecurityDialog()

' Open ZPACK transaction
session.findById("wnd[0]").maximize
session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
session.findById("wnd[0]").sendVKey 0

' Enter material number and execute search
session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber
session.findById("wnd[0]").sendVKey 8

' Get the file list table
Set fileTable = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/cntlCONTROL_9103_2/shellcont/shell")

' Initialize variables for finding files
found = False
foundDPI = False
rowIndex = -1
dpiRowIndex = -1
foundFileName = ""

' Loop through the files to find TRP and DPI files
For i = 0 To fileTable.RowCount - 1
    fileName = fileTable.GetCellValue(i, "DOKNR") ' Document number column
    
    ' Check for TRP (PICT) files
    If InStr(1, fileName, "PICT", vbTextCompare) > 0 Then
        found = True
        rowIndex = i
        foundFileName = fileName
        Exit For
    ' Check for DPI files if TRP not found yet
    ElseIf InStr(1, fileName, "DPI", vbTextCompare) > 0 Then
        foundDPI = True
        dpiRowIndex = i
        ' Don't exit, continue looking for PICT which has priority
    End If
Next

' Handle TRP file if found
If found Then
    ' If TRP file found, select it
    fileTable.setCurrentCell rowIndex, "DOKNR" ' Adjust column ID if needed
    fileTable.selectedRows = rowIndex
    fileTable.doubleClickCurrentCell
    
    ' Process the TRP document
    ProcessDocument foundFileName, "TRP"
    
' Handle DPI file if found and TRP not found
ElseIf foundDPI Then
    ' If only DPI file found, open it instead
    fileTable.setCurrentCell dpiRowIndex, "DOKNR" ' Adjust column ID if needed
    fileTable.selectedRows = dpiRowIndex
    fileTable.doubleClickCurrentCell
    
    ' Process the DPI document
    ProcessDocument "DPI", "DPI"
Else
    ' Return nothing found status
    LogMessage "NOT-FOUND"
End If

' Navigate back in SAP
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press

' Close the log file at the end of script
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

CloseLogFile()